/*
 * Title:         树表采用进度条加载处理
 * Description:  [描述文件的功能、作用、使用方法和注意事项]
 * Copyright:     Copyright (c) 1988-2015
 * Company:       Huawei Tech. Co., Ltd
 * @author         j65716
 * @version        版本  2010-4-19
 */

//进度条处理对象
var g_progressLoadTreeTbl = null;

var TimeOperation =
{    
    /**
     * 用setInterval调用对象的函数
     * 
     * @param obj 回调的目标对象
     * @param func 回调的函数
     * @param args 回调传递的参数
     * @param intervalTime 超时时间
     * 
     */
    doObjSetInterval : function(obj, func, args, intervalTime)
    {
        return setInterval( function()
        {
            func.apply(obj, args);
        }, intervalTime);
    }
}

/**
 * 进度条加载树表处理类，每个页面需要进行注册
 * @author 蒋英茹 65716
 * @since iManager U2000 V1R2,2010-4-22
 * @param parentGridName 进度条父容器的id
 * @param progressBarObj 进度条对象
 * @return
 */
function ProgressLoadTreeTbl(parentGridName, progressBarObj)
{
    this._parentGridName = parentGridName;
    this._progressBarObj = progressBarObj;
    this._insertRowIndex = 0;
    this._insertIntervalID = null;
    this._rowsNum = 0;
    
    //是否为IE浏览器
    this._isIEBrowser = (navigator.appName.indexOf("Microsoft") != -1);
    
    //进度条加载每次插入的行数量
    this._progressAddRowNum = 20;
    
    //启用进度条方式加载的阈值，即大于该值则启用进度条方式加载，否则不启用：IE定位60，firefox定为100
    this._maxRowNumProgress = this._isIEBrowser ? 30 : 80;
}

/**
 * 采用一定的策略计算每次插入的数目
 * @author 蒋英茹 65716
 * @return
 */
ProgressLoadTreeTbl.prototype.getProgressAddRowNum = function()
{
    if(this._isIEBrowser)
    {
        //500条以内，每次插入10%
        if(this._rowsNum < 300)
        {
            this._progressAddRowNum = parseInt(this._rowsNum/10)
        }
        else
        {
            //大于500行，为防止过慢，最多插入50
            this._progressAddRowNum = 30;
        }
    }
    else
    {
        //500条以内，每次插入10%
        if(this._rowsNum < 500)
        {
            this._progressAddRowNum = parseInt(this._rowsNum/10)
        }
        else
        {
            //大于500行，最多插入50
            this._progressAddRowNum = 50;
        }
    }   
}


/**
 * 处理添加所有行操作, 
 * 各页面需要定义 插入行方法addRowDataFunc(rowData, index, margs) 
 *               插入后的处理：overProcFunc(margs);
 *               进度条加载前，进行限制：limitBeforeLoadTable()
 *               加载完成，取消限制： cancelLimitAfterLoadTable()
 * @author 蒋英茹 65716
 * @since iManager U2000 V1R2,2010-4-21
 * @param isProgressLoad 是否采用进度条加载
 * @param rowsNum 行数
 * @param rowsData 插入行数据
 * @param addRowDataFunc 插入行操作方法（函数指针）
 * @param overProcFunc 加载完毕后的回调方法（注意是函数指针）
 * @param margs 各个操作要传递的参数列表
 * @param colsNum 树表的列数目（普通配置表格均为3列，可不传，默认为3；批量的为批量个数+2）
 * @return
 */
ProgressLoadTreeTbl.prototype.progressLoadTotalRows = function(isProgressLoad, rowsNum, rowsData, addRowDataFunc, overProcFunc, margs)
{
    //初始化信息
    this._insertRowIndex = 0;
    this._rowsNum = rowsNum;
            
    
    if (!isProgressLoad || (rowsNum <= this._maxRowNumProgress))
    {
        // 隐藏进度条
        this.showProgressBar(false);

        for ( var i = 0; i < rowsNum; i++)
        {
            // 插入行数据,个页面自己处理，增加该方法
            addRowDataFunc.apply(this, [ rowsData[i], i, margs ]);
        }

        if (null != overProcFunc)
        {
            // 调用各个页面自定义的剩余处理方法otherInfoHandle
            // otherInfoHandle(margs);
            overProcFunc.apply(this, [ margs ]);
        }

    }
    else
    {
        // 进度条加载前，进行限制，由各个页面自己定义
        limitBeforeLoadTable();

        //计算每次插入的数码
        this.getProgressAddRowNum();

        //alert("this._rowsNum=" + this._rowsNum + " this._progressAddRowNum=" + this._progressAddRowNum + " this._maxRowNumProgress=" + this._maxRowNumProgress);
        
        // logger.log("enter addTableData showProgressBar insertData");
        // 显示进度条
        this.showProgressBar(true);
        // logger.log("exit addTableData showProgressBar insertData");
        // logger.log("enter addTableData insertData");

        // 插入行数据，并且更新进度条上的百分比显示,定时器全局管理，只有一个
        this._insertIntervalID = TimeOperation.doObjSetInterval(this, this._insertData, [ rowsNum, rowsData,
                addRowDataFunc, overProcFunc, margs ], 1);
    }
}

/**
 * 插入数据处理，对外不关注，内部使用
 * 
 * @author 蒋英茹 65716
 * @since iManager U2000 V1R2,2010-4-21
 * @param rowNum 总行数
 * @param allRowData 插入行数据
 * @param parentGridName 父容器id
 * @param addRowDataFunc 插入行操作方法（函数指针）
 * @param overProcFunc 加载完毕后的回调方法（注意是函数指针）
 * @param margs 参数列表
 * @return
 */
ProgressLoadTreeTbl.prototype._insertData = function(rowNum, allRowData, addRowDataFunc, overProcFunc, margs)
{
    // 每次插入多行
    for ( var i = 0; i < this._progressAddRowNum; i++)
    {
        addRowDataFunc.apply(this, [ allRowData[this._insertRowIndex], this._insertRowIndex, margs ]);
        // addRowData(allRowData[g_insertRowIndex], g_insertRowIndex, margs);
        this._insertRowIndex++;

        // 停止定时操作
        if (this._insertRowIndex == rowNum)
        {
            // logger.log("exit addTableData insertData");
            // 停止定时任务，隐藏进度条
            this.stopRowInterval();

            this.updateProgressbar(1, "");

            this.showProgressBar(false);
            // logger.log("enter addTableData otherInfoHandle");

            // 加载完成，取消限制，由各个页面自己定义
            cancelLimitAfterLoadTable();

            if (null != overProcFunc)
            {
                // 调用各个页面自定义的剩余处理方法otherInfoHandle
                // otherInfoHandle(margs);
                overProcFunc.apply(this, [ margs ]);
            }

            return;
        }
    }

    if (this._insertRowIndex < rowNum)
    {
        // 更新进度条的显示
        this.updateProgressbar((this._insertRowIndex / rowNum), "");
    }
}

/**
 * 是否显示进度条
 * 
 * @param isShow 为true时显示进度条，为false时不显示进度条
 * @return
 */
ProgressLoadTreeTbl.prototype.showProgressBar = function(isShow)
{
    var box = window.xui.getWidget("ProgressbarBox");
    var gridData = box.getLayoutData();

    var labelBox = window.xui.getWidget("LabelBox");
    var labelGridData = labelBox.getLayoutData();

    if (isShow)
    {
        labelGridData.expandY = false;
        labelGridData.preferHeight = 0;

        labelGridData.paddingX = 10;
        labelGridData.paddingY = 0;

        labelBox.setLayoutData(labelGridData);

        gridData.expandY = false;
        gridData.preferHeight = 20;

        gridData.paddingX = 10;
        gridData.paddingY = 5;

        box.setLayoutData(gridData);
    }
    else
    {
        gridData.expandY = false;
        gridData.preferHeight = 0;

        gridData.paddingX = 10;
        gridData.paddingY = 0;

        box.setLayoutData(gridData);

        labelGridData.expandY = false;
        labelGridData.preferHeight = 20;

        labelGridData.paddingX = 10;
        labelGridData.paddingY = 3;

        labelBox.setLayoutData(labelGridData);
    }
    var rootGrid = window.xui.getWidget(this._parentGridName);
    rootGrid.relayout();
}

/**
 * 更新进度条显示
 * 
 * @param value
 * @param content
 * @return
 */
ProgressLoadTreeTbl.prototype.updateProgressbar = function(value, content)
{
    if (value >= 1)
    {
        value = 1;
        content = "";
    }

    this._progressBarObj.update(value, Math.floor(value * 100) + "% " + content);
}

/**
 * 停止定时任务
 */
ProgressLoadTreeTbl.prototype.stopRowInterval = function()
{
    if (this._insertIntervalID != null)
    {
        clearInterval(this._insertIntervalID);
        this._insertIntervalID = null;
    }
}

/**
 * 判断是否正在加载行数据
 */
ProgressLoadTreeTbl.prototype.isLoadingRowData = function()
{
    if (this._insertIntervalID != null)
    {
        return true;
    }
    
    return false;

}

/**
 * 这个方法可以用于声明两个类之间的继承关系
 * 
 * @param subClass 子类名称
 * @param superClass 父类名称
 */
function extend(subClass, superClass)
{
    var F = function()
    {
    };
    F.prototype = superClass.prototype;
    subClass.prototype = new F();
    subClass.prototype.constrctor = subClass;

    subClass.superclass = superClass.prototype;
    if (superClass.prototype.constructor == Object.prototype.constructor)
    {
        superClass.prototype.constructor = superClass;
    }
}

/**
 * 批量页面的进度条处理类（主要是最大行数不同，批量列数较多，每行的插入速度会比较慢）
 * @author 蒋英茹 65716
 * @since iManager U2000 V1R2,2010-4-22
 * @param parentGridName
 * @param progressBarObj
 * @param _colsNum value的列数
 * @return
 */
function ProgressLoadBatchTreeTbl(parentGridName, progressBarObj, colsNum)
{
    ProgressLoadBatchTreeTbl.superclass.constructor.call(this, parentGridName, progressBarObj);
    
    //树表列数
    this._colsNum = colsNum;
    
    //获取最大临界采用进度条的行数量
    this._getMaxRowNumProgress();
}

extend(ProgressLoadBatchTreeTbl, ProgressLoadTreeTbl);

/**
 * 根据浏览器以及列数决定是否采用进度条加载（浏览器以及列数不同，树表插入的性能不同）
 * @author 蒋英茹 65716
 * @return
 */
ProgressLoadTreeTbl.prototype._getMaxRowNumProgress = function()
{
    if(this._isIEBrowser)
    {
        //5列作为分界点
        if(this._colsNum < 4)
        {
            this._maxRowNumProgress = 20
        }
        else
        {
            this._maxRowNumProgress = 10
        }        
    }
    else
    {
      //5列作为分界点
        if(this._colsNum < 8)
        {
            this._maxRowNumProgress = 30
        }
        else
        {
            this._maxRowNumProgress = 15
        }   
    }
}


/**
 * 采用一定的策略计算每次插入的数目
 * @author 蒋英茹 65716
 * @return
 */
ProgressLoadBatchTreeTbl.prototype.getProgressAddRowNum = function()
{
    if(this._isIEBrowser)
    {
        //在3列，采用基类算法
        if(this._colsNum < 4)
        {
            ProgressLoadBatchTreeTbl.superclass.getProgressAddRowNum.call(this);
        }
        else
        {
            //5列以上每次插入的步长固定为3，不能太多（这个数量不会过多,IE性能比较差）
            this._progressAddRowNum = 3;
        }
    }
    else
    {
        //在7列以内，采用基类算法
        if(this._colsNum < 8)
        {
            ProgressLoadBatchTreeTbl.superclass.getProgressAddRowNum.call(this);
        }
        else
        {
            //5列以上每次插入的步长固定为5，不能太多（这个数量不会过多）
            this._progressAddRowNum = 5;
        }
    }
    

}


